
**** Stata replication code for Clemens, 'Partial Legalization and the Parallel Market'
**** Working Paper April 2024

version 18.0
frames reset
clear all
graph set window fontface "Linux Libertine O" 
set more off

*** Running this code requires the preliminary steps of 1) ensuring the necessary .ado files are installed, and 2) replacing the filepath in global 'working' with the path to the directory where you have stored the data files 

*** 1. INSTALL THE FOLLOWING .ADO FILES

* Requires -sankey- by Asjad Naqvi, and -palette- package by Ben Jann 
* https://github.com/asjadnaqvi/stata-sankey

* Requires -geoplot- and -geoframe- by Ben Jann
* net install geoplot, replace from(https://raw.githubusercontent.com/benjann/geoplot/main/)
* net install geoframe, replace from(https://raw.githubusercontent.com/benjann/geoplot/main/)

* Requires -moremata-, -colrspace-, and -palettes- packages 
* ssc install palettes, replace
* ssc install colrspace, replace
* ssc install moremata, replace

*** 2. REPLACE THE X'S BELOW WITH THE PATH TO YOUR WORKING DIRECTORY

global working = "XXXXXXXXXX"

global rawdata = "$working"
global data = "$working"
global output = "$working"

frame rename default main


*****
***** FIGURES 3-10 AND TABLES 2-3
*****

* Import data 

use "$data/encounters_dispositions.dta", clear

gen ln_unlawful_entry = ln(unlawful_entry)

gen ln_unlawful_entry_mx = ln(unlawful_entry_mex)    // Can't write _mex due to variable name length limit for -lpirf- 

gen ln_lawful_released = ln(lawful_released)
gen ln_lawful_released_n = ln(lawful_released-ofo_detain)

gen ln_unlawful_released = ln(unlawful_released)
gen ln_unlawful_released_n = ln(unlawful_released-usbp_detain)

* Get total migrant encounters from October 2011 through July 2023
gen enc_all = encountercount_trac
replace enc_all = enc_new_usbp + enc_new_ofo if inrange(date,`=tm(2018m10)',`=tm(2023m7)')
egen tot = total(enc_all)
format %12.0gc tot
list tot in 160
drop enc_all tot

* Figure 3

graph twoway line ln_unlawful_entry ln_lawful_released date if inrange(date,`=tm(2011m10)',`=tm(2023m7)'), lcolor(cranberry midgreen) ///
	scheme(s1color) aspect(0.5) ///
	xtitle("Calendar date, {it:monthly}", margin(medium) size(*.85)) ///
	ytitle("Encounters or releases per month, {it:log scale}", margin(medsmall) size(*.85)) ///
	xlab(`=tm(2011m1)' `=tm(2012m1)' `=tm(2013m1)' `=tm(2014m1)' `=tm(2015m1)' `=tm(2016m1)' `=tm(2017m1)' `=tm(2018m1)' `=tm(2019m1)' `=tm(2020m1)' `=tm(2021m1)' `=tm(2022m1)'  `=tm(2023m1)' ,format(%tmCY) labsize(*.7)  grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(`=tm(2011m1)'(1)`=tm(2023m7)', tlcolor(black) tlwidth(vthin)) ///
	ylabel(`=ln(100)' "100"  `=ln(1000)' "1,000" `=ln(10000)' "10,000" `=ln(50000)' "50,000" `=ln(100000)' "100,000" `=ln(200000)' "200,000"  `=ln(300000)' "300,000",  labsize(*.7) angle(0) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(`=ln(100)' `=ln(200)' `=ln(300)' `=ln(400)' `=ln(500)' `=ln(600)' `=ln(700)' `=ln(800)' `=ln(900)' ///
	`=ln(2000)' `=ln(3000)' `=ln(4000)' `=ln(5000)' `=ln(6000)' `=ln(7000)' `=ln(8000)' `=ln(9000)' ///
	`=ln(20000)' `=ln(30000)' `=ln(40000)' `=ln(50000)' `=ln(60000)' `=ln(70000)' `=ln(80000)' `=ln(90000)' ///
	`=ln(110000)' `=ln(120000)' `=ln(130000)' `=ln(140000)' `=ln(150000)' `=ln(160000)' `=ln(170000)' `=ln(180000)' `=ln(190000)' ///
	`=ln(210000)' `=ln(220000)' `=ln(230000)' `=ln(240000)' `=ln(250000)' `=ln(260000)' `=ln(270000)' `=ln(280000)' `=ln(290000)', tlcolor(black) tlwidth(vthin)) ///
	graphregion(margin(r+20)) plotregion(margin(*1 *1 *1)) legend(off) ///
	text(`=ln(150000)' `=tm(2023m10)' "Unlawful crossings" "encountered", color(cranberry) justification(left) placement(east) size(*.7)) ///
	text(`=ln(43000)' `=tm(2023m10)' "Releases after" "lawful crossing", color(midgreen) justification(left) placement(east) size(*.7)) ///
	 plotregion(lalign(outside))	// Prevents gridlines from overlapping frame 

graph export "$output/overview.pdf", replace


* Figure 4

qui: reg ln_unlawful_entry ln_lawful_released, robust
mat result=r(table)
scalar b = result[1,1]
local b3 : di %04.3f scalar(b)
di "`b3'"
scalar se = result[2,1]
local se3 : di %04.3f scalar(se)
di "`se3'"

graph twoway lfitci  ln_unlawful_entry ln_lawful_released  if inrange(date,`=tm(2011m10)',`=tm(2023m7)'), estopts(robust) lcolor(edkblue)  fcolor(ebg) alwidth(0) lwidth(*.75)  ///
	|| scatter ln_unlawful_entry ln_lawful_released  if inrange(date,`=tm(2011m10)',`=tm(2023m7)'), mlcolor(black) mlwidth(vthin) mfcolor(black%15) ///
	scheme(s1color) aspect(1) ///
	xtitle("Releases after lawful crossing, {it:log scale}", margin(medium) size(*1)) ///
	ytitle("Unlawful crossings encountered, {it:log scale}", margin(medsmall) size(*1)) ///
	xlabel(`=ln(100)' "100"  `=ln(1000)' "1,000" `=ln(10000)' "10,000" `=ln(50000)' "50,000" ,  labsize(*1) angle(0) grid glcolor(gs13) glwidth(vthin) gmax gmin ) ///
	xmtick(`=ln(100)' `=ln(200)' `=ln(300)' `=ln(400)' `=ln(500)' `=ln(600)' `=ln(700)' `=ln(800)' `=ln(900)' ///
	`=ln(2000)' `=ln(3000)' `=ln(4000)' `=ln(5000)' `=ln(6000)' `=ln(7000)' `=ln(8000)' `=ln(9000)' ///
	`=ln(20000)' `=ln(30000)' `=ln(40000)' `=ln(50000)' `=ln(60000)' `=ln(70000)' , tlcolor(black) tlwidth(vthin)) ///
	ylabel(`=ln(10000)'  "10,000" `=ln(50000)' "50,000" `=ln(100000)' "100,000" `=ln(200000)' "200,000"  `=ln(300000)' "300,000",  labsize(*1) angle(0) grid glcolor(gs13) glwidth(vthin) gmax gmin) ///
	ymtick(`=ln(8000)' `=ln(9000)' ///
	`=ln(20000)' `=ln(30000)' `=ln(40000)' `=ln(50000)' `=ln(60000)' `=ln(70000)' `=ln(80000)' `=ln(90000)' ///
	`=ln(110000)' `=ln(120000)' `=ln(130000)' `=ln(140000)' `=ln(150000)' `=ln(160000)' `=ln(170000)' `=ln(180000)' `=ln(190000)' ///
	`=ln(210000)' `=ln(220000)' `=ln(230000)' `=ln(240000)' `=ln(250000)' `=ln(260000)' `=ln(270000)' `=ln(280000)' `=ln(290000)', tlcolor(black) tlwidth(vthin)) ///
	legend(off)  plotregion(lalign(outside)) ///
	text(`=ln(280000)' `=ln(600)' "Elasticity `b3'", placement(west) justification(right) size(*.75) color(emidblue)) ///
	text(`=ln(245000)' `=ln(600)' "s.e. `se3'", placement(west) justification(right) size(*.75) color(emidblue))
	
graph export "$output/naive_levels.pdf", replace	
	
* Generate difference variables because Stata lfitci cannot take 'd.' operator
tsset date 
gen d_ln_unlawful_entry = d.ln_unlawful_entry
gen d_ln_lawful_released = d.ln_lawful_released
	
* Store regression results for display in figure
qui: reg d_ln_unlawful_entry d_ln_lawful_released, robust
mat result=r(table)
scalar b = result[1,1]
local b3 : di %04.3f scalar(b)
di "`b3'"
scalar se = result[2,1]
local se3 : di %04.3f scalar(se)
di "`se3'"	
	
graph twoway lfitci d_ln_unlawful_entry d_ln_lawful_released if inrange(date,`=tm(2011m10)',`=tm(2023m7)'), estopts(robust) lcolor(edkblue) fcolor(ebg) alwidth(0) lwidth(*.75) ///
	|| scatter d_ln_unlawful_entry d_ln_lawful_released if inrange(date,`=tm(2011m10)',`=tm(2023m7)'), mlcolor(black) mlwidth(vthin) mfcolor(black%15) ///
	scheme(s1color) aspect(1) ///
	xtitle("{&Delta} releases after lawful crossing, {it:log points}", margin(medium) size(*1)) ///
	ytitle("{&Delta} unlawful crossings encountered, {it:log points}", margin(medsmall) size(*1)) ///
	legend(off) ///
	xlabel(-2(1)1, format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) xmtick(-2.5(.1)1.5, tlwidth(vthin)) ylabel(-.5(.5).5, angle(0) format(%02.1f) grid glcolor(gs13)  glwidth(vthin) gmax gmin) ymtick(-.5(.1).5, tlwidth(vthin)) plotregion(lalign(outside)) ///
	text(.535 -1.47 "Elasticity `b3'", placement(west) justification(right) size(*.75) color(emidblue)) ///
	text(.485 -1.47 "s.e. `se3'", placement(west) justification(right) size(*.75) color(emidblue))
	
graph export "$output/naive_changes.pdf", replace	
	
		
* Figure 5

lpirf ln_unlawful_entry ln_lawful_released, lags(1/6) vce(robust) step(12)
irf set myirfs.irf, replace 
irf create myirf

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_lawful_released" & response=="ln_unlawful_entry"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(midblue%40) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(midblue%66) ///
	|| line lower_ci step, lcolor(midblue%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Lawful released, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Unlawful crossing", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-.5(.5)1.5,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-.7(.1)1.7)  plotregion(lalign(outside))
graph save "$output/irf_1.gph", replace
restore	

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_unlawful_entry" & response=="ln_unlawful_entry"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(gray%55) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(gray%66) ///
	|| line lower_ci step, lcolor(gray%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Unlawful crossing, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Unlawful crossing", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-.5(.5)1.5,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-.7(.1)1.7) plotregion(lalign(outside))
graph save "$output/irf_2.gph", replace
restore	

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_lawful_released" & response=="ln_lawful_released"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(gray%55) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(gray%66) ///
	|| line lower_ci step, lcolor(gray%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Lawful released, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Lawful released", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-1(1)2,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-1.2(.1)2.4) plotregion(lalign(outside))
graph save "$output/irf_3.gph", replace
restore	
	

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_unlawful_entry" & response=="ln_lawful_released"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(gray%55) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(gray%66) ///
	|| line lower_ci step, lcolor(gray%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Unlawful crossing, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Lawful released", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-1(1)2,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-1.2(.1)2.4) plotregion(lalign(outside))
graph save "$output/irf_4.gph", replace
restore	
	
graph combine "$output/irf_1.gph" "$output/irf_2.gph" "$output/irf_3.gph" "$output/irf_4.gph", rows(2) xsize(3.2) ysize(3.9) 

graph export "$output/irf_full_lp.pdf", replace
	
	
* Figure 6 and Table 2

capture program drop irf_graph
program define irf_graph
	args lags
	preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_lawful_released" & response=="ln_unlawful_entry"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(midblue%40) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(midblue%66) ///
	|| line lower_ci step, lcolor(midblue%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Lawful" "released ({it:ln}), `lags' lags", margin(small) size(*.8)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(medsmall) size(*1.2)) ///
	ytitle("Response: Unlawful crossing ({it:ln})", margin(medsmall) size(*1.2)) ///
	xlab(0(2)12, grid glcolor(gs13) glwidth(vthin) gmax gmin labsize(*1.2))  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-.8(.2).2, angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin labsize(*1.2)) ymtick(-.7(.2).1) plotregion(lalign(outside))
graph save "$output/irf`lags'.gph", replace
restore
end

* (a) Using Local Projections

lpirf ln_unlawful_entry ln_lawful_released, lags(1/3) vce(robust) step(12)		// This gives Table 2 col. 1
irf set myirfs.irf, replace 
irf create myirf
irf table irf, std noci
est store LP3
irf_graph 3

lpirf ln_unlawful_entry ln_lawful_released, lags(1/6) vce(robust) step(12)		// This gives Table 2 col. 2
irf set myirfs.irf, replace 
irf create myirf
irf table irf, std noci
est store LP6
irf_graph 6

lpirf ln_unlawful_entry ln_lawful_released, lags(1/12) vce(robust) step(12)		// This gives Table 2 col. 3
irf set myirfs.irf, replace 
irf create myirf
irf table irf, std noci
est store LP12
irf_graph 12


graph combine "$output/irf3.gph" "$output/irf6.gph" "$output/irf12.gph", rows(1) ycommon
graph export "$output/irf1_lp.pdf", replace

est table LP3 LP6 LP12, b(%04.3f) se(%04.3f)

* (b) Using VAR

var ln_unlawful_entry ln_lawful_released, lags(1/3)  		// This gives Table 2 col. 4
irf create myirf, set(myirfs) replace step(12)
irf table irf, std noci				
irf_graph 3

var ln_unlawful_entry ln_lawful_released, lags(1/6) 		// This gives Table 2 col. 5
irf create myirf, set(myirfs) replace step(12)
irf table irf, std noci
irf_graph 6

var ln_unlawful_entry ln_lawful_released, lags(1/12) 		// This gives Table 2 col. 6
irf create myirf, set(myirfs) replace step(12)
irf table irf, std noci
irf_graph 12


graph combine "$output/irf3.gph" "$output/irf6.gph" "$output/irf12.gph", rows(1) ycommon
graph export "$output/irf1_var.pdf", replace

	

* Figure 7

lpirf ln_unlawful_entry ln_lawful_released ln_unlawful_released, lags(1/6) vce(robust) step(12)
irf set myirfs.irf, replace 
irf create myirf

* Row 1

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_lawful_released" & response=="ln_unlawful_entry"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(midblue%40) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(midblue%66) ///
	|| line lower_ci step, lcolor(midblue%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Lawful released, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Unlawful crossing", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-1(.5)2.5,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-1(.1)2.5) plotregion(lalign(outside))
graph save "$output/irf_1.gph", replace
restore	

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_unlawful_released" & response=="ln_unlawful_entry"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(gray%55) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(gray%66) ///
	|| line lower_ci step, lcolor(gray%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Unlawful released, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Unlawful crossing", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-1(.5)2.5,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-1(.1)2.5) plotregion(lalign(outside))
graph save "$output/irf_2.gph", replace
restore	

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_unlawful_entry" & response=="ln_unlawful_entry"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(gray%55) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(gray%66) ///
	|| line lower_ci step, lcolor(gray%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Unlawful crossing, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Unlawful crossing", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-1(.5)2.5,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-1(.1)2.5) plotregion(lalign(outside))
graph save "$output/irf_3.gph", replace
restore	

graph combine "$output/irf_1.gph" "$output/irf_2.gph" "$output/irf_3.gph", rows(1) 
graph export "$output/irf_3way_row1.pdf", replace

* Row 2

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_lawful_released" & response=="ln_unlawful_released"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(gray%55) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(gray%66) ///
	|| line lower_ci step, lcolor(gray%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Lawful released, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Unlawful released", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-1(1)4.4,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-1.2(.1)4.4) plotregion(lalign(outside))
graph save "$output/irf_1.gph", replace
restore	

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_unlawful_released" & response=="ln_unlawful_released"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(gray%55) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(gray%66) ///
	|| line lower_ci step, lcolor(gray%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Unlawful released, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Unlawful released", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-1(1)4.4,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-1.2(.1)4.4) plotregion(lalign(outside))
graph save "$output/irf_2.gph", replace
restore	

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_unlawful_entry" & response=="ln_unlawful_released"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(gray%55) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(gray%66) ///
	|| line lower_ci step, lcolor(gray%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Unlawful crossing, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Unlawful released", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-1(1)4.4,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-1.2(.1)4.4) plotregion(lalign(outside))
graph save "$output/irf_3.gph", replace
restore	

graph combine "$output/irf_1.gph" "$output/irf_2.gph" "$output/irf_3.gph", rows(1) 
graph export "$output/irf_3way_row2.pdf", replace

* Row 3

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_lawful_released" & response=="ln_lawful_released"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(gray%55) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(gray%66) ///
	|| line lower_ci step, lcolor(gray%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Lawful released, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Lawful released", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-2(1)3,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-2.2(.1)3.2) plotregion(lalign(outside))
graph save "$output/irf_1.gph", replace
restore	

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_unlawful_released" & response=="ln_lawful_released"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(gray%55) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(gray%66) ///
	|| line lower_ci step, lcolor(gray%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Unlawful released, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Lawful released", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-2(1)3,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-2.2(.1)3.2) plotregion(lalign(outside))
graph save "$output/irf_2.gph", replace
restore	

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_unlawful_entry" & response=="ln_lawful_released"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(gray%55) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(gray%66) ///
	|| line lower_ci step, lcolor(gray%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Unlawful crossing, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Lawful released", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-2(1)3,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-2.2(.1)3.2) plotregion(lalign(outside))
graph save "$output/irf_3.gph", replace
restore	

graph combine "$output/irf_1.gph" "$output/irf_2.gph" "$output/irf_3.gph", rows(1) 
graph export "$output/irf_3way_row3.pdf", replace


* Figure 8 

lpirf ln_unlawful_entry ln_lawful_released ln_unlawful_released, lags(1/6) vce(robust) step(12) 
irf set myirfs.irf, replace 
irf create myirf
irf_graph 6

graph export "$output/irf1_lp_usbp_rel.pdf", replace

lpirf ln_unlawful_entry ln_lawful_released, lags(1/6) vce(robust) step(12) exog(_Imon_2-_Imon_12)
irf set myirfs.irf, replace 
irf create myirf
irf_graph 6

graph export "$output/irf1_lp_monthI.pdf", replace

forvalues i = 1(1)6 {
	gen l`i'_unemp_hispanic = l`i'.unemp_hispanic  // lpirf's exog cannot accept time-series operators, so must make these variables manually 
}

lpirf ln_unlawful_entry ln_lawful_released, lags(1/6) vce(robust) step(12) exog(l1_unemp_hispanic-l6_unemp_hispanic)
irf set myirfs.irf, replace 
irf create myirf
irf_graph 6

graph export "$output/irf1_lp_pull.pdf", replace


lpirf ln_unlawful_entry ln_lawful_released ln_unlawful_released, lags(1/6) vce(robust) step(12) exog(_Imon_2-_Imon_12 l1_unemp_hispanic-l6_unemp_hispanic)
irf set myirfs.irf, replace 
irf create myirf
irf_graph 6

graph export "$output/irf1_lp_cumul.pdf", replace

lpirf ln_unlawful_entry ln_lawful_released if date>=`=ym(2017,1)', lags(1/6) vce(robust) step(12)
irf set myirfs.irf, replace 
irf create myirf
irf_graph 6

graph export "$output/irf1_lp_posttrump.pdf", replace

lpirf ln_unlawful_entry ln_lawful_released if !inrange(date,`=ym(2020,3)',`=ym(2020,8)'), lags(1/6) vce(robust) step(12)  // By Sept 2020, USBP apprehensions were 1) above same month in previous year and 2) moreoever above USBP apprehensions at SWB in any September since 2006: https://www.cbp.gov/sites/default/files/assets/documents/2020-Jan/U.S.%20Border%20Patrol%20Monthly%20Apprehensions%20%28FY%202000%20-%20FY%202019%29_1.pdf
irf set myirfs.irf, replace 
irf create myirf
irf_graph 6

graph export "$output/irf1_lp_nocovid.pdf", replace


* Figure 9

qui: summ unemp_hispanic
sca unemp_mean = r(mean)
gen unemp_hisp_demean = unemp_hispanic-unemp_mean 
qui: summ unemp_hisp_demean, detail
sca unemp_low = r(p10)	// Store Hispanic unemployment at 10th percentile
sca unemp_high = r(p90)	// Store Hispanic unemployment at 90th percentile

matrix het = (.,.,.,.,.\.,.,.,.,.\.,.,.,.,.\.,.,.,.,.\.,.,.,.,.\.,.,.,.,.\.,.,.,.,.\.,.,.,.,.\.,.,.,.,.\.,.,.,.,.\.,.,.,.,.\.,.,.,.,.\.,.,.,.,.)  // 13x5

forvalues i = 1(1)13 {
	mat het[`i',1] = `i'-1  // First row is month zero
	
	reg f`=`i'-1'.ln_unlawful_entry c.l1.ln_lawful_released##c.l1.unemp_hisp_demean l(2/6).ln_lawful_released l(2/6).unemp_hisp_demean l(1/6).ln_unlawful_entry , vce(robust)
	
	mat het[`i',2] = _b[l1.ln_lawful_released] + 0*_b[cL.ln_lawful_released#cL.unemp_hisp_demean]	// Store LP estimate at mean unemployment
	mat het[`i',3] = _se[l1.ln_lawful_released] 	// Store LP s.e. at mean unemployment
	
	mat het[`i',4] = _b[l1.ln_lawful_released] + unemp_low*_b[cL.ln_lawful_released#cL.unemp_hisp_demean]	// Store LP estimate at low unemployment
	mat het[`i',5] = _b[l1.ln_lawful_released] + unemp_high*_b[cL.ln_lawful_released#cL.unemp_hisp_demean]	// Store LP estimate at high unemployment
}

svmat het
ren het1 time
ren het2 lp_mean
gen lp_mean_u = lp_mean + 1.96*het3
gen lp_mean_l = lp_mean - 1.96*het3
ren het4 lp_low_unemp
ren het5 lp_high_unemp

graph twoway rarea lp_mean_u lp_mean_l time, fcolor(midblue%30) lwidth(0) ///
	|| line lp_mean time, lcolor(midblue%47) lwidth(*1.5) ///
	|| line lp_mean_u time, lcolor(midblue%32) lwidth(*.33) ///
	|| line lp_mean_l time, lcolor(midblue%32) lwidth(*.33) ///
	|| line lp_high_unemp lp_low_unemp time, lcolor(black black) lpattern(vshortdash dash) lwidth(*1.7 *1.7) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Lawful" "released ({it:ln}), `lags' lags", margin(small) size(*.8)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(medsmall) size(*1.2)) ///
	ytitle("Response: Unlawful crossing ({it:ln})", margin(medsmall) size(*1.2)) ///
	xlab(0(2)12, grid glcolor(gs13) glwidth(vthin) gmax gmin labsize(*1.2))  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-.8(.2).2, angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin labsize(*1.2)) ymtick(-.7(.2).1) plotregion(lalign(outside)) ///
	graphregion(margin(r+60)) ///
	text(-0.35 12.75 "Hispanic unemployment 10th pctile: 4.2", color(black) placement(east) justification(left)) ///
	text(-0.5 12.75 "Average Hispanic unemployment: 7.1", color(midblue%55) placement(east) justification(left)) ///
	text(-0.7 12.75 "Hispanic unemployment 90th pctile: 11.1", color(black) placement(east) justification(left)) 
	
graph export "$output/heterogeneous.pdf", replace

	
* Table 3

sureg (f(0/5).ln_unlawful_entry = l(1/6).ln_unlawful_entry l(1/6).ln_lawful_released) if e(sample), vce(robust) dfk small // Coeff on first lag in each of these is the LP IRF at each horizon 1-6

* Get IRF coefficients
lpirf ln_unlawful_entry ln_lawful_released, lags(1/6) vce(robust) step(12)
mat result = r(table)
forvalues x = 1/12 {
	dis result[1,`=24+`x'']
	sca irf_coeff_`x' = result[1,`=24+`x'']
}

* Get shocks to lawful_released
reg ln_lawful_released L(1/6).ln_unlawful_entry L(1/6).ln_lawful_released, vce(robust)
predict rel_shock, resid
forvalues x = 1/12 {
	gen shock_effect_`x' = l`x'.rel_shock * scalar(irf_coeff_`x')
}
egen shock_effect = rowtotal(shock_effect_1-shock_effect_12)

corr ln_unlawful_entry shock_effect, cov
dis r(cov_12)/r(Var_1)

** Now repeat with bootstrapped standard errors

capture program drop myratio
program myratio, rclass
        version 18
		qui: corr ln_unlawful_entry shock_effect, cov
		local rat = r(cov_12)/r(Var_1)
        return scalar ratio = `rat'
end

bootstrap r(ratio), seed(314159265) reps(10000): myratio

drop shock_effect* 	// So that it can be calculated again below

** Include unlawful released 

* Get IRF coefficients
lpirf ln_unlawful_entry ln_lawful_released ln_unlawful_released, lags(1/6) vce(robust) step(12)
mat result = r(table)
forvalues x = 1/12 {
	dis result[1,`=36+`x'']
	sca irf_coeff_`x' = result[1,`=36+`x'']
}

* Get shocks to lawful_released
reg ln_lawful_released L(1/6).ln_unlawful_entry L(1/6).ln_lawful_released L(1/6).ln_unlawful_released, vce(robust)
predict rel_shock2, resid
forvalues x = 1/12 {
	gen shock_effect_`x' = l`x'.rel_shock2 * scalar(irf_coeff_`x')
}
egen shock_effect = rowtotal(shock_effect_1-shock_effect_12)

bootstrap r(ratio), seed(314159265) reps(10000): myratio

drop shock_effect* 	// So that it can be calculated again below

** Include unlawful released AND add month fixed effects and Hispanic unemployment 

* Get IRF coefficients
lpirf ln_unlawful_entry ln_lawful_released ln_unlawful_released, lags(1/6) vce(robust) step(12) exog(_Imon_2-_Imon_12 l1_unemp_hispanic-l6_unemp_hispanic)
mat result = r(table)
forvalues x = 1/12 {
	dis result[1,`=36+`x'']
	sca irf_coeff_`x' = result[1,`=36+`x'']
}

* Get shocks to lawful_released
reg ln_lawful_released L(1/6).ln_unlawful_entry L(1/6).ln_lawful_released L(1/6).ln_unlawful_released _Imon_2-_Imon_12 l1_unemp_hispanic-l6_unemp_hispanic, vce(robust) 
predict rel_shock3, resid
forvalues x = 1/12 {
	gen shock_effect_`x' = l`x'.rel_shock3 * scalar(irf_coeff_`x')
}
egen shock_effect = rowtotal(shock_effect_1-shock_effect_12)

bootstrap r(ratio), seed(314159265) reps(10000): myratio


* Figure 10

gen overall = lawful_released+ unlawful_entry
gen lawful_frac_util = lawful_released/overall
replace lawful_frac_util = . if date!=`=tm(2018m4)'
egen lawful_frac = mean(lawful_frac_util)
gen lawful_counterfac = lawful_frac * overall
gen unlawful_counterfac = (1-lawful_frac)*overall

graph twoway line lawful_released lawful_counterfac unlawful_entry unlawful_counterfac date if inrange(date,`=tm(2017m7)',`=tm(2018m12)'), ///
	xline(`=tm(2018m4)', lpattern(vshortdash) lcolor(black%33)) ///
	lcolor(emerald emerald%45 maroon maroon%45) lpattern(solid shortdash solid shortdash) ylab(10000(10000)50000, format(%9.0gc)) ysc(log) ymtick(2000(1000)50000) ///
	xlab(`=tm(2017m7)' `" "Jul." "2017" "' `=tm(2018m1)' `" "Jan." "2018" "' `=tm(2018m7)' `" "Jul." "2018" "' `=tm(2019m1)' `" "Jan." "2019" "',format(%tmCY) labsize(*1)  grid glcolor(gs13) glpattern(solid) glwidth(0) gmax gmin) ///
	aspect(1.1) ///
	xmtick(`=tm(2017m7)'(1)`=tm(2018m12)', tlcolor(black) tlwidth(vthin)) ytitle("Encounters per month, {it: log scale}") xtitle("Date (monthly)", margin(medium)) legend(off) ///
	text(59000 `=tm(2019m1)-.5' "Unlawful" "crossing", color(maroon) placement(east) justification(left)) ///
	text(38500 `=tm(2019m1)-.5' "{it:Fixed April}" "{it:2018 frac.}", color(maroon%45) placement(east) justification(left)) ///
	text(7700 `=tm(2019m1)-.5'  "Lawful" "releases", color(emerald) placement(east) justification(left))  ///
	text(12500 `=tm(2019m1)-.5' "{it:Fixed April}" "{it:2018 frac.}", color(emerald%45) placement(east) justification(left))  ///
	text(67000 `=tm(2018m4)' "Metering", color(black%33) justification(center)) ///
	graphregion(margin(t+10))

graph export "$output/experiment2018square.pdf", replace

gen gap_unlawful = ln(unlawful_entry)-ln(unlawful_counterfac)
gen gap_lawful = ln(lawful_released)-ln(lawful_counterfac)
summ gap_lawful if date==`=tm(2018m6)'
sca base = r(mean)
gen effect = gap_unlawful/base
gen graphdate = date-`=tm(2018m4)'
graph twoway line effect graphdate if inrange(date,`=tm(2018m4)',`=tm(2019m1)'), lcolor(black) lwidth(*1.25) ///
	|| line effect graphdate if inrange(date,`=tm(2019m1)',`=tm(2019m4)'), lcolor(black%20) lpattern(vshortdash) ///
	scheme(s1color) aspect(1) legend(off)  ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse ({it:0 = April 2018})", margin(medsmall) size(*1.2)) ///
	ytitle("Response: Unlawful crossing ({it:ln})", margin(medsmall) size(*1.2)) ///
	xlab(0(2)12, grid glcolor(gs13) glwidth(vthin) gmax gmin labsize(*1.2))  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-.8(.2).2, angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin labsize(*1.2)) ymtick(-.7(.2).1) plotregion(lalign(outside))

graph export "$output/experiment2018scaled.pdf", replace


* Appendix Table A1

eststo sumstats: quietly estpost sum ///
ln_unlawful_entry ln_lawful_released ln_unlawful_released unemp_hispanic, detail 

esttab sumstats using "$output/summary_stats.tex", cells("mean(fmt(%04.3f))  sd(fmt(%04.3f)) min(fmt(%04.3f)) max(fmt(%04.3f)) count(fmt(%4.0f))") nonumbers coeflabels(ln_unlawful_entry "ln Unlawful crossing" ln_lawful_released "ln Lawful released"  ln_unlawful_released "ln Unlawful released" unemp_hispanic "Unemployment (Hisp./Latino)")replace 


* Appendix Figure A2

xcorr ln_lawful_released ln_lawful_released, lags(20) ytitle("Cross corr. with {it:ln} lawful release") xtitle("Lag of {it:ln} lawful release") ylabel(,format(%02.1f) axis(1))  ylabel(,format(%02.1f) axis(2)) aspect(.33) title("") xmtick(-20(1)20) ymtick(-1(.1)1, axis(1))  ymtick(-1(.1)1, axis(2))
graph export "$output/xcorr_lawful_lawful.pdf", replace

xcorr ln_unlawful_entry ln_unlawful_entry, lags(20) ytitle("Cross corr. with {it:ln} unlawful crossing") xtitle("Lag of {it:ln} unlawful crossing") ylabel(,format(%02.1f) axis(1))  ylabel(,format(%02.1f) axis(2)) aspect(.33) title("") xmtick(-20(1)20) ymtick(-1(.1)1, axis(1))  ymtick(-1(.1)1, axis(2))
graph export "$output/xcorr_unlawful_unlawful.pdf", replace

xcorr ln_lawful_released ln_unlawful_entry, lags(20) ytitle("Cross corr. with {it:ln} unlawful crossing") xtitle("Lag of {it:ln} lawful release") ylabel(,format(%02.1f) axis(1))  ylabel(,format(%02.1f) axis(2)) aspect(.33) title("") xmtick(-20(1)20) ymtick(-1(.1)1, axis(1))  ymtick(-1(.1)1, axis(2))
graph export "$output/xcorr_lawful_unlawful.pdf", replace


* Appendix Figure A6

lpirf ln_unlawful_entry_mx ln_lawful_released, lags(1/6) vce(robust) step(12)
irf set myirfs.irf, replace 
irf create myirf

preserve		// For full control over IRF graph, create by hand 
	use myirfs.irf, clear
	keep if impulse=="ln_lawful_released" & response=="ln_unlawful_entry_mx"
	keep step irf stdirf
	gen upper_ci= irf +1.96*stdirf
	gen lower_ci = irf - 1.96*stdirf
	graph twoway rarea upper_ci lower_ci step, fcolor(midblue%40) lwidth(0) ///
	|| line irf step, lcolor(black) lwidth(*1.25) ///
	|| line upper_ci step, lcolor(midblue%66) ///
	|| line lower_ci step, lcolor(midblue%66) ///
	scheme(s1color) aspect(1) legend(off) title("Impulse: Lawful released, 6 lags", margin(small) size(*.75)) ///
	yline(0, lcolor(cranberry) lpattern(vshortdash) lwidth(medium)) ///
	xtitle("Months since impulse", margin(small) size(*1)) ///
	ytitle("Response: Unlawful crossing", margin(medsmall) size(*1)) ///
	xlab(0(2)12,labsize(*1) grid glcolor(gs13) glwidth(vthin) gmax gmin)  ///
	xmtick(1(2)11, tlcolor(black%20)) ///
	ylab(-.8(.2).2,angle(0) format(%02.1f) grid glcolor(gs13) glwidth(vthin) gmax gmin) ymtick(-.7(.2).1) plotregion(lalign(outside))
	graph export "$output/irf1_lp_mex.pdf", replace
restore	




*****
***** Figure 2
*****

frames reset
clear all

* Level 1

frame rename default main

use "$data/encounters_dispositions.dta", clear

replace lawful_released_narrow_new = lawful_released_narrow-ofo_detain if inrange(date,`=tm(2020m10)',`=tm(2022m4)') // During this date range (and all others after June 2018) the TRAC data don't report OFO NTAs separately by detention status. But the number of detentions of OFO encounters with credible fear is reported on the CBP website. Thus fill in where possible by combining the two series

keep if inrange(date,`=tm(2020m10)',`=tm(2023m4)') // Keep the date range for which ICE detention final releases are available and current

frame copy main level1
frame copy main level2
frame change level1

collapse (sum) enc_new_usbp enc_new_ofo
gen x=_n
reshape long enc_new_, i(x) j(component) string
*egen source=total(enc_new_)
ren enc_new_ encounters
ren component destination  
gen source = "all"
drop x
order source destination encounters
gen layer = 1

sankey encounters, from(source) to(destination) by(layer)

* Level 2

frame change level2

collapse (sum)  enc_new_usbp enc_new_ofo lawful_released_narrow_new ofo_detain unlawful_released usbp_detain

ren lawful_released_narrow_new release_ofo
gen release_usbp = unlawful_released - usbp_detain
drop unlawful_released	// This 'broad' definition includes detentions 
ren ofo_detain detain_ofo
ren usbp_detain detain_usbp

ren enc_new_ofo encounters_ofo
ren enc_new_usbp encounters_usbp

gen x=_n
reshape long encounters_ release_ detain_, i(x) j(component) string

ren component source
gen no_access_ = encounters_ - (detain_ + release_)
drop encounters_

ren detain_ result1
ren release_ result2
ren no_access_ result3 

keep source result1 result2 result3
reshape long result, i(source) j(destination) string

ren result encounters
replace destination = "detained" if destination=="1"
replace destination = "released" if destination=="2"
replace destination = "no access" if destination=="3"

gen layer = 2

save "$data/level2_temp.dta", replace
frame change level1
append using "$data/level2_temp.dta"
erase "$data/level2_temp.dta"

order source destination encounters

sankey encounters, from(source) to(destination) by(layer)

* Store number of migrants detained, for third level

egen detained = total(encounters) if destination=="detained"
summ detained
scalar define det = r(mean)
dis scalar(det)
drop detained 

* Level 3

frame create level3
frame change level3

use "$data/ice_detention.dta", clear

keep if inrange(date,`=tm(2020m10)',`=tm(2023m4)')
collapse (sum) cbp_book_ins out_cbp_est
gen frac = out_cbp_est/cbp_book_ins
summ frac
scalar define frac_released = r(mean)

clear 
set obs 2
gen layer = 3
gen source = "detained"
gen destination = "released"
replace destination = "no access" if _n==2
gen encounters = .
 
replace encounters = round(scalar(det) * scalar(frac_released)) if destination == "released" 
replace encounters = round(scalar(det) * (1-scalar(frac_released))) if destination == "no access"

save "$data/level3_temp.dta", replace
frame change level1
append using "$data/level3_temp.dta"
erase "$data/level3_temp.dta"

* Set up labels to that -sankey- shows categories in custom order
gen source2 = .
gen destination2 = .

foreach x in source destination {
	replace `x'2 = 0 if `x'=="all"
	replace `x'2 = 1 if `x'=="ofo"
	replace `x'2 = 2 if `x'=="usbp"
	replace `x'2 = 3 if `x'=="released"
	replace `x'2 = 4 if `x'=="detained"
	replace `x'2 = 5 if `x'=="no access"
}

cap drop trace2
gen trace2 = .
replace trace2 = 2 if source2==2 & destination2==3 & layer==2
replace trace2 = 3 if source2==1 & destination2==3 & layer==2
replace trace2 = 2 if source2==2 & destination2==4 & layer==2
replace trace2 = 3 if source2==1 & destination2==4 & layer==2
* replace trace2 = 3 if source2==4 & destination2==3 & layer==3

* Translate absolute encounter numbers into fractions
egen tot_enc = total(encounters) if source=="all"
summ tot_enc 
scalar define tot = r(mean)
gen enc_frac = round(encounters / scalar(tot), 0.001)
format enc_frac %04.3f

lab de labels 0 "All encounters, Oct. 2020–Apr. 2023" 1 "Lawful" 2 "Unlawful" 3 "{it:Released}" 4 "Detained" 5 "No access", replace

lab val source2 labels
lab val destination2 labels

sankey enc_frac, from(source2) to(destination2) by(layer) ///
	 boxwid(4.2) colorvar(trace2) palette(s2) sort1(name) sort2(order) ///
	  novalleft xsize(*.3) ysize(*.55) gap(11) labsize(2.35) labcolor(white) valsize(2) labgap(60) format(%04.3f) ///
	  offset(10) smooth(2.5) colorvarmiss(gs9) colorboxmiss(gs8) 

	*	  ctitles("To border" "Crossing" `" "Initial" "disposition" "'  "Final disposition") ctsize(2.5) // ctgap(-5)
	  
	*  graphregion(margin(b+10)) ///

graph export "$output/sankey_orig.pdf", replace 
graph save "$output/sankey_orig.gph", replace 


*****
***** Additional Appendix Figures and Tables
*****

* Appendix Figure A1

frames reset
clear all
frame rename default main

* Import main dataset

use "$data/encounters_dispositions.dta", clear

gen year=yofd(dofm(date))
gen fy=year
gen month=month(dofm(date))
tab month
replace fy=fy+1 if inrange(month,10,12)
list date month year fy in 1/50

* Overall releases by fiscal year 

gen released = lawful_released+ unlawful_released
collapse (sum) released, by(fy)
replace released = . if fy==2023 // Currently does not include entire fiscal year
replace released = . if released==0

* Get EOIR data on asylum filings

frame create eoir
frame change eoir
import excel "$rawdata/14_total_asylum_applications.xlsx", sheet("Table 1") cellrange(B10:C26) firstrow
ren FiscalYear fy
ren Filed asylum_filings

frame change main
frlink 1:1 fy, frame(eoir) generate(link)	
frget asylum_filings, from(link)
drop link

* Ratio of persons covered to number of cases

dis 340587/238841  // FY 2022 Table 6a,6c in: https://www.dhs.gov/sites/default/files/2023-11/2023_0818_plcy_refugees_and_asylees_fy2022.pdf
dis 91052/63074  // FY 2021 Table 6a,6c in: https://www.dhs.gov/sites/default/files/2023-11/2023_0818_plcy_refugees_and_asylees_fy2022.pdf
dis 143999/93778  // FY 2020 Table 6a,6c in: https://www.dhs.gov/sites/default/files/2023-11/2023_0818_plcy_refugees_and_asylees_fy2022.pdf

gen upper = 1.55 * asylum_filings
gen lower = 1.35 * asylum_filings 

gen fy_shift = fy-1	// compare releases to asylum filings one year later
drop if fy_shift<2011
drop if fy_shift==2023

graph twoway line released fy, lcolor(maroon) || line asylum_filings fy_shift, lcolor(midgreen) lpattern(dash) || rarea upper lower fy_shift, fcolor(midgreen%50) alwidth(0) ///
	xlabel(2012(2)2022) xmtick(2012(1)2022) ytitle("Number of people {it:or} filings") xtitle("Fiscal year") ///
	legend(label(1 "{it:People}: Overall releases") label(2 "{it:Filings}: Asylum applications {bf:one year later}") label(3 "{it:People}: Estimated people covered by asylum filings {bf:one year later}") position(6)) plotregion(margin(0)) ylabel(0(500000)1000000, format(%12.0gc)) ymtick(0(100000)1200000) 
	
graph export "$output/asylum_filings.pdf", replace


* Appendix Figure A3(a)

use "$data/ice_detention.dta", clear

graph twoway scatter cbp_book_ins out_cbp_est date, c(l l) clcolor(cranberry midgreen) msize(*.33 *.33) mcolor(cranberry midgreen) ///
	xline(`=tm(2020m10)', lcolor(gs12)) xline(`=tm(2023m4)', lcolor(gs12)) ///
	xtitle("Calendar date, {it:monthly}", margin(medium) size(*1)) ///
	ytitle("Number of book-ins" "or final releases", margin(medsmall) size(*1)) ///
	xlab(`=tm(2019m1)' `=tm(2020m1)' `=tm(2021m1)' `=tm(2022m1)'  `=tm(2023m1)' ,format(%tmCY) labsize(*.8)) ylabel(, angle(0) format(%08.0gc)) ymtick(0(1000)50000) ///
	xmtick(`=tm(2018m9)'(1)`=tm(2023m10)', tlcolor(black%20)) scheme(s1color) plotregion(margin(none)) aspect(.4) legend(off) ///
	text(31000 `=tm(2022m2)' "Book-ins", color(cranberry) placement(east) justification(left)) ///
	text(6200 `=tm(2022m2)' "Final releases", color(midgreen) placement(east) justification(left))

graph export "$output/recent_releases.pdf", replace 


* Appendix Figure A3(b)

graph twoway scatter avg_length_cbp date, c(l) clcolor(midblue) msize(*.33) mcolor(midblue) ///
	xtitle("Calendar date, {it:monthly}", margin(medium) size(*1)) ///
	ytitle("Average length of stay (days)," "CBP transfers", margin(medsmall) size(*1)) ///
	xlab(`=tm(2019m1)' `=tm(2020m1)' `=tm(2021m1)' `=tm(2022m1)'  `=tm(2023m1)' ,format(%tmCY) labsize(*.8)) ///
	ylabel(0(50)150, angle(0) format(%08.0gc)) ymtick(0(10)150) ///
	xmtick(`=tm(2018m9)'(1)`=tm(2023m10)', tlcolor(black%20)) scheme(s1color) plotregion(margin(none)) aspect(.4) legend(off) 

graph export "$output/length_stay.pdf", replace 

* Appendix Figure A3(c)

use "$data/encounters_dispositions.dta", clear

merge 1:1 date using "$data/ice_detention.dta"

* For some dates, CBP does not publish statistics on the number of initial releases by OFO. I indirectly estimate this by the following procedure
* 1) estimate the number of USBP detentions
gen usbp_detain_est = unlawful_released-unlawful_released_narrow
replace usbp_detain_est= usbp_detain if usbp_detain_est==.

* 2) For years where the OFO fraction of all CBP detentions is observed, this fraction is generally in the range of 0.04--0.12
gen ofo_frac = ofo_detain/(ofo_detain+ usbp_detain)
qui summ ofo_frac
sca ofo_frac_mean = r(mean)

* 3) Thus estimate OFO detentions as the same fraction of all CBP book-ins to ICE 
gen ofo_detain_hi = 0.12 * cbp_book_ins
gen ofo_detain_mid = ofo_frac_mean * cbp_book_ins
gen ofo_detain_lo = 0.02 * cbp_book_ins 

* 4) And thus estimate the fraction of initial releases (not detained) as a fraction of all OFO encounters that remain in US
gen frac_est_hi = (lawful_released - ofo_detain_hi)/lawful_released if date!=`=tm(2020m4)'		// Method gives nonsense values in the month of total border closure
gen frac_est_mid = (lawful_released - ofo_detain_mid)/lawful_released if date!=`=tm(2020m4)'	// Method gives nonsense values in the month of total border closure
gen frac_est_lo = (lawful_released - ofo_detain_lo)/lawful_released if date!=`=tm(2020m4)'		// Method gives nonsense values in the month of total border closure

summ frac_est_mid if inrange(date,`=tm(2018m7)',`=tm(2020m9)')

* For some dates, this fraction can be directly observed because CBP does not publish statistics on the number of initial releases by OFO specifically:

gen lawful_released_initial = lawful_released_narrow if date<=`=tm(2018m6)'
replace lawful_released_initial = lawful_released-ofo_detain if date>=`=tm(2020m10)'
gen ln_lawful_released_i = ln(lawful_released_initial)

gen frac = lawful_released_initial/lawful_released

graph twoway scatter frac date if date>=`=tm(2011m10)' & date<=`=tm(2018m6)', c(l) msize(*.3) mcolor(purple) lcolor(purple) ///
	|| scatter frac date if date<=`=tm(2023m7)' & date>=`=tm(2020m10)',  c(l) msize(*.3) mcolor(purple) lcolor(purple) ///
	|| rarea frac_est_hi frac_est_lo date if inrange(date,`=tm(2018m7)',`=tm(2020m9)'),  fcolor(black%25) lwidth(0) ///
	|| scatter frac_est_mid date if inrange(date,`=tm(2018m7)',`=tm(2020m9)'),  c(l) msize(*.3) mcolor(gs9) lcolor(gs9) ///
	ylabel(0(.2)1) legend(off) xtitle("Calendar date, {it:monthly}", margin(medium) size(*1)) ///
	ytitle("Initial releases," "fraction of all lawful crossing", margin(medsmall) size(*1)) ///
	xlab(`=tm(2011m1)' `=tm(2012m1)' `=tm(2013m1)' `=tm(2014m1)' `=tm(2015m1)' `=tm(2016m1)' `=tm(2017m1)' `=tm(2018m1)' `=tm(2019m1)' `=tm(2020m1)' `=tm(2021m1)' `=tm(2022m1)'  `=tm(2023m1)' ,format(%tmCY) labsize(*.8)) ylabel(, angle(0) format(%02.1f)) ymtick(0(.1)1) ///
	xmtick(`=tm(2011m1)'(1)`=tm(2023m8)', tlcolor(black%20)) scheme(s1color) plotregion(margin(none)) aspect(.4) ///
	text(0.4 `=tm(2017m1)' "Actual", placement(east) justification(left) color(purple)) ///
	text(0.25 `=tm(2018m11)' "{it:Estimated}", placement(east) justification(left) color(gs9)) 
	
graph export "$output/initial_releases_ofo.pdf", replace 


* Appendix Figures A4-A5

clear
import excel "$rawdata/table 6 analysis.xlsx", sheet("main") cellrange(A16:F26) firstrow

gen frac = Removedbyanotheragency/Encounters 
summ frac

drop if FiscalYear>2019	// The data source used here does not include Title 42 expulsions, so this analysis is less useful for 2019 and afterward, for which the Appendix includes analysis based on more direct data 

lab var Removedorreturned "of which removed or returned"
lab var Removed "of which removed"
lab var Removedbyanotheragency "of which removed by other agency"
lab var Fraction "Other agency removals as frac. of OFO releases"

graph twoway area Encounters Removedorreturned Removed Removedbyanotheragency  FiscalYear, color(gs10 midblue emerald maroon) ///
	aspect(.6) ylabel(0(50000)150000,format(%9.0gc)) ymtick(0(10000)170000) xlabel(2013(1)2019) 
	
graph export "$output/ofo_removals.pdf", replace
graph twoway scatter Fraction FiscalYear, c(l)  xlabel(2013(1)2019) ylabel(0(.2)1, format(%02.1f)) ymtick(0(.1)1) aspect(1)
graph export "$output/ofo_removal_frac.pdf", replace
mean Fraction


***** 
***** Table 1
*****

frames reset
clear

* USBP 

use "$data/encounters_dispositions.dta", clear
keep if inrange(date,`=tm(2011m10)',`=tm(2023m7)')
collapse (sum) unlawful_entry
list														// Table 1 row 1

* Count OFO fraud cases

use "$rawdata/extract_ofo.dta", clear

* Create date variable: Note that 'fymonth' variable, notwithstanding the name, contains the *calendar* date with month 
tostring fymon, gen(date_string)
gen year_string = substr(date_string,1,4)
gen month_string = substr(date_string,5,2)
destring year_string, gen(year)
destring month_string, gen(month)
gen date = ym(year,month)
format date %tm
keep if date<=`=tm(2022m5)'

* Keep only encounters at the Southwest border, that is, in an OFO location responsible for ports of entry at border with Mexico
keep if inlist(field_office_dsp,"Laredo","El Paso","Tucson","San Diego")  // See page 16 of https://www.cbp.gov/sites/default/files/assets/documents/2017-Mar/FY-2016-CBP-PAR-508C.pdf
 
* Standardize variable names with CBP Public Data Portal and collapse individual level data to encounter counts by observed traits 
gen component = "ofo"
replace family = "Single Adult" if family=="Unknown"
rename family demographic
ren citizenship_dsp citizenshipgrouping
gen titleofauthority = "Title 8"
replace titleofauthority = "Title 42" if entry_status_grp == "Title 42"
gen encountercount_trac=1

gen unlawful_entry=0
replace unlawful_entry = 1 if component=="ofo" & (inlist(entry_status_grp,"False or Fraudulent Claim","Smuggler") | entry_status_dsp=="Overstay")

collapse (sum) unlawful_entry  // at Ports of Entry

list														// Table 1 row 2

* OFO
* Updated TRAC OFO data May 2022 through July 2023

frame create new_ofo_disp_trac
frame change new_ofo_disp_trac

import excel "$rawdata/trac ofo july 2023 update.xlsx", sheet("Sheet2") cellrange(A6:K66) firstrow clear
drop F-K
split date, parse(-)
destring date1, replace
destring date2, replace
drop date
gen date = ym(date1, date2)
format date %tm
drop date1 date2

drop if date==`=tm(2022m5)'	// Don't double-count May 2022
collapse (sum) nta paroled exped_remov_cred_fear
list

summ exped_remov_cred_fear
sca ex_rem_cred_fear2 = r(mean)

summ nta
sca nta2 = r(mean)

summ paroled
sca paroled2 = r(mean)

* Import TRAC data for OFO encounters through May 2022

frame create trac_ofo
frame change trac_ofo 

use "$rawdata/extract_ofo.dta", clear

* Create date variable: Note that 'fymonth' variable, notwithstanding the name, contains the *calendar* date with month 
tostring fymon, gen(date_string)
gen year_string = substr(date_string,1,4)
gen month_string = substr(date_string,5,2)
destring year_string, gen(year)
destring month_string, gen(month)
gen date = ym(year,month)
format date %tm
keep if date<=`=tm(2022m5)'

* Keep only encounters at the Southwest border, that is, in an OFO location responsible for ports of entry at border with Mexico
keep if inlist(field_office_dsp,"Laredo","El Paso","Tucson","San Diego")  // See page 16 of https://www.cbp.gov/sites/default/files/assets/documents/2017-Mar/FY-2016-CBP-PAR-508C.pdf
 
* Standardize variable names with CBP Public Data Portal and collapse individual level data to encounter counts by observed traits 
gen component = "ofo"
replace family = "Single Adult" if family=="Unknown"
rename family demographic
ren citizenship_dsp citizenshipgrouping
gen titleofauthority = "Title 8"
replace titleofauthority = "Title 42" if entry_status_grp == "Title 42"
gen encountercount_trac=1

collapse (sum) encountercount_trac, by(date component entry_status_grp entry_status_dsp ofo_disposition_grp ofo_disposition_dsp citizenshipgrouping)

tab ofo_disposition_grp
collapse (sum) encountercount_trac, by(ofo_disposition_grp)
list

summ encountercount_trac if ofo_disposition_grp=="Expedited Removal - Credible Fear"
sca ex_rem_cred_fear1 = r(mean)

summ encountercount_trac if ofo_disposition_grp=="Notice to Appear"
sca nta1 = r(mean)

summ encountercount_trac if ofo_disposition_grp=="Paroled"
sca paroled1 = r(mean)

summ encountercount_trac if ofo_disposition_grp=="Prosecutorial Discretion"
sca discretion1 = r(mean)

*** Calculate table values, Oct. 2011 through July 2023 	// Table 1 rows 3-6
dis ex_rem_cred_fear1+ex_rem_cred_fear2		// Expedited removal-Credible fear
dis nta1+nta2								// Notice to appear
dis paroled1+paroled2						// Paroled
dis discretion1+0							// Prosecutorial discretion

** New CBP data, post TRAC

frame create new_cbp_dispositions
frame change new_cbp_dispositions

import excel "$rawdata/latest dispositions through august 2023.xlsx", sheet("Sheet1") cellrange(A5:AK53) firstrow clear 
drop if component == "" | component=="component"
reshape long m, i(component type) j(date)
ren m count
format date %tm
order date component type count
compress
sort date component type
destring count, replace

keep if component=="usbp_disposition"

drop if date<=`=tm(2022m5)'	// Don't double-count May 2022 and before
drop if date>`=tm(2023m7)'	// Don't include dates for USBP that aren't available for OFO

collapse (sum) count, by(type)
list 

summ count if type=="Warrant/Notice To Appear (NTA) - Detained"
sca nta2 = r(mean)

summ count if type=="Notice To Appear/Order of Recognizance, I-385 – Released"
sca nta5 = r(mean)

summ count if type=="Parole + ATD"
sca parole1 = r(mean)




collapse (sum) count
list

* Import TRAC data for USBP encounters through May 2022

frame create trac_usbp
frame change trac_usbp

use "$rawdata/extract_bp.dta"

* Create date variable: Note that 'fymonth' variable, notwithstanding the name, contains the *calendar* date with month 
tostring fymon, gen(date_string)
gen year_string = substr(date_string,1,4)
gen month_string = substr(date_string,5,2)
destring year_string, gen(year)
destring month_string, gen(month)
gen date = ym(year,month)
format date %tm
keep if date<=`=tm(2022m5)'
drop arrest_date-date_string year_string year month_string month

* Keep only encounters at the Southwest border, that is, in Office of Border Patrol sectors adjacent to the border with Mexico 
drop if inlist(sector_dsp,"Blaine Sector","Buffalo Sector","Detroit Sector","Grand Forks Sector","Havre Sector")
drop if inlist(sector_dsp,"Houlton Sector","Miami Sector","New Orleans Sector","Ramey Sector","Spokane Sector","Swanton Sector")
keep if inlist(arrest_method_dsp, "Patrol Border", "Title 42")	// Drop all encounters via local law enforcement or others in the interior of the sector 

* Standardize variable names with CBP Public Data Portal and collapse individual level data to encounter counts by observed traits 
gen component = "usbp"
rename family demographic
ren citizenship_dsp citizenshipgrouping
ren arrest_method_dsp titleofauthority
gen encountercount_trac=1
replace demographic = "Other" if demographic=="Unknown"
collapse (sum) encountercount_trac, by(date component entry_status_dsp disposition_dsp citizenshipgrouping)

drop if date<tm(2011m10)	// Since OFO data only extend back to October 2011, USBP data before that date are not useful for estimating total encounters 
collapse (sum) encountercount_trac, by(disposition_dsp)

list

summ encountercount_trac if disposition_dsp=="Notice to Appear (warrant of arrest)"
sca nta1 = r(mean)

summ encountercount_trac if disposition_dsp=="Notice to Appear (detained I-862)"
sca nta3 = r(mean)

summ encountercount_trac if disposition_dsp=="Notice to Appear (released I-861)"
sca nta4 = r(mean)

summ encountercount_trac if disposition_dsp=="Expedited Removal/Credible Fear"
sca ercf1 = r(mean)

summ encountercount_trac if disposition_dsp=="Paroled"
sca parole2 = r(mean)

summ encountercount_trac if disposition_dsp=="Reinstate/Reasonable Fear"
sca reinstate1 = r(mean)

summ encountercount_trac if disposition_dsp=="Released"
sca released1 = r(mean)


dis nta1+nta2+nta3		// NTA detained					// Table 1 rows 7-12 
dis nta4+nta5			// NTA released
dis ercf1+	0				// Expedited removal/Credible fear 
dis parole1+	parole2			// Paroled
dis reinstate1 +	0				// Reinstate/Reasonable fear 
dis released1 +0				// Released 



*****
***** Figure 1 map
*****

frames reset
clear all
frame rename default main

* Estimate relative numbers of apprehensions in each sector 
frame create apprehensions
frame change apprehensions
use "$data/all_encounters.dta", clear
drop if date>`=tm(2022m5)'
drop if source=="portal"
keep if component=="usbp"
mdesc encountercount_trac
collapse (sum) encountercount_trac, by(sector_dsp)
egen enc_sum = total(encountercount_trac)
gen enc_frac = encountercount_trac/enc_sum
gen enc_rest = 1-enc_frac 
gen enc_pct = enc_frac*100
rename sector_dsp bord_pat_s   // for matching with map data 
replace bord_pat_s = subinstr(bord_pat_s," Sector", "", .)
save "$data/appre_by_sector.dta", replace

* USBP sectors

frame change main 
cd "$rawdata/map files/united_states_border_patrol_sectors"
shp2dta using united_states_border_patrol_sectors, ///
	gencentroids(centroid) genid(unit_id) ///
	data("usbp_sectors_data") coor("usbp_sectors_coordinates") replace
use "usbp_sectors_data", clear
merge 1:1 bord_pat_s using "$data/appre_by_sector.dta"
save "usbp_sectors_data.dta", replace 
geoframe create sectors usbp_sectors_data.dta, id(unit_id) shpfile(usbp_sectors_coordinates.dta) // coord(_X _Y)

geoframe create centers usbp_sectors_data.dta, id(unit_id) coord(x_centroid y_centroid)

* US and Mexico boundaries

cd "$rawdata/map files/natural earth"
*spshp2dta using ne_50m_admin_0_countries, data("ne_50m_admin_0_countries_data")  coor("ne_50m_admin_0_countries_coordinates")

spshape2dta ne_50m_admin_0_map_subunits, replace
geoframe create earth ne_50m_admin_0_map_subunits.dta

* Roads

cd "$rawdata/map files/north american environmental atlas/Roads_Shapefile/NA_Roads/data"
spshape2dta roads_l_v2, replace
geoframe create roads roads_l_v2.dta

* Ports of Entry

cd "$rawdata/map files/DOT" // As of August 1, 2023
spshape2dta geo_export_e56ad27c-f323-40de-9773-cb1d39e49aa1, replace
geoframe create poe geo_export_e56ad27c-f323-40de-9773-cb1d39e49aa1.dta
keep if date==mdy(8,1,2023)
keep if border=="US-Mexico Border"
collapse (first) _ID _CX _CY, by(port_name state latitude longitude)

* Some of the labels overlap, so offset selected ones
gen port_name2 = ""
replace port_name2 = port_name if inlist(port_name,"San Ysidro","Roma")
gen port_name3 = ""
replace port_name3 = port_name if inlist(port_name,"San Luis","Tecate")
replace port_name = "" if port_name2 != "" | port_name3 != "" // clear duplicates
* Combine some labels for clarity
replace port_name = "Otay Mesa, CBX" if port_name=="Otay Mesa"
replace port_name = "" if port_name=="Cross Border Xpress"
replace port_name = "" if port_name=="Tecate"
replace port_name = "Calexico, Calexico East" if port_name=="Calexico"
replace port_name = "" if port_name=="Calexico East"
replace port_name = "Santa Teresa, El Paso" if port_name=="Santa Teresa"
replace port_name = "" if port_name=="El Paso"

* Reposition sector labels for clarity

frame change centers

replace y_centroid = y_centroid-.5 if bord_pat_s=="Rio Grande Valley"	// Exact positioning of bubbles on final map

replace x_centroid = x_centroid-1 if bord_pat_s=="Big Bend"
replace y_centroid = y_centroid-1.2 if bord_pat_s=="Big Bend"

replace y_centroid = y_centroid-.6 if bord_pat_s=="El Paso"

replace x_centroid = x_centroid-.25 if bord_pat_s=="Del Rio"

replace x_centroid = x_centroid-1.6 if bord_pat_s=="Laredo"
replace y_centroid = y_centroid-1.3 if bord_pat_s=="Laredo"

replace x_centroid = x_centroid+1.7 if bord_pat_s=="Yuma"
replace y_centroid = y_centroid-3.7 if bord_pat_s=="Yuma"

replace x_centroid = x_centroid+3 if bord_pat_s=="El Centro"
replace y_centroid = y_centroid-3 if bord_pat_s=="El Centro"

replace x_centroid = x_centroid+6.6 if bord_pat_s=="San Diego"
replace y_centroid = y_centroid-1.5 if bord_pat_s=="San Diego"

* Draw map

geoplot (area earth if SUBUNIT=="United States", fcolor("237 221 222") lcolor(red%50)) ///
	(line roads if TYPE<3 & inlist(COUNTRY,"USA") & _CX>-130, lcolor(brown*.40) lwidth(*.33)) ///
	(line sectors if !inlist(unit_id,2,17), lcolor(red%50)) ///
	(area earth if SUBUNIT=="Mexico", fcolor(brown*.25) lcolor(black)) ///
	(line roads if TYPE<3 & inlist(COUNTRY,"MEX") & _CX>-130, lcolor(brown*.43) lwidth(*.33)) ///
	(line earth if SUBUNIT=="Mexico", lcolor(black)) ///
	(point poe, msymbol(diamond) color(white) mlcolor(none) msize(*.66) mlwidth(0)) ///
	(point poe, msymbol(diamond) color(green%45) mlcolor(black) msize(*.66) mlwidth(*.33)) ///
	(label poe port_name, color(green%75) position(8) angle(40) size(*.35) mlabgap(*.7)) ///
	(label poe port_name2, color(green%75) position(9) angle(40) size(*.35) mlabgap(*.7)) ///
	(label poe port_name3, color(green%75) position(7) angle(40) size(*.35) mlabgap(*.7)) ///
  	(point centers [w=enc_frac] if !inlist(unit_id,2), mfcolor(red%50) mlcolor(none) mlabel() msize(*.8)) ///
	, plotregion(color(midblue%33)) ///
	text(25 -105 "MEXICO", color(gray%50) justification(center) size(*.4)) ///
	text(35.7	-111.0821	"TUCSON", color(red%50) size(*.4)) ///
	text(36.20212	-114.4081	"YUMA", color(red%50) size(*.4)) ///
	text(35.1	-106.1088	"EL PASO", color(red%50) size(*.4)) ///
	text(33.93221	-101.5122	"BIG" "BEND", color(red%50) size(*.4)) ///
	text(27.74375	-95.48966	"RIO" "GRANDE" "VALLEY", color(red%50) size(*.4) justification(left) placement(east)) ///
	text(30.8147	-96.30807	"LAREDO", color(red%50) size(*.4)) ///
	text(31.88187	-98.90928	"DEL" "RIO", color(red%50) size(*.4)) ///
	text(34.91196	-121.0078	"SAN DIEGO", color(red%50) size(*.4)) ///
	text(36.00	-117.5652	"EL" "CENTRO", color(red%50) size(*.4)) 

	*  	(point centers if !inlist(unit_id,2), mcolor(none) mlabel(bord_pat_s) mlabcolor(red%50) mlabpos(12) mlabsize(*.5)) ///
	
graph export "$output/border_map.pdf", replace 	
